﻿
//
// AddSubdivisions.js
//
function addSubdivisions(geom, polyCollection, polyIndex, splitEdgesMap, polysToSelect) {

    var polygonPointCount = geom.getPolygonPointCount(polyIndex);
    if (polygonPointCount < 3) {
        return;
    }
    // services.debug.trace("splitting poly : " + polyIndex);

    var containingMesh = polyCollection.getContainingMesh();

    // determine if we need to add new texture coordinates
    var IndexingModeUndefined = 0;
    var IndexingModePerPoint = 1;
    var IndexingModePerPointOnPolygon = 3;

    // determine the material index on the pre division polygon
    var materialIndex = geom.getPolygonMaterialIndex(polyIndex);

    // this is our starting index for new points
    var newPointStart = geom.pointCount;

    // we'll compute the center of the triangle
    var avgPos = [0, 0, 0];
    var avgTex = [0, 0];

    // iterate over polypoints and total
    for (var i = 0; i < polygonPointCount; i++) {
        var point = geom.getPointOnPolygon(polyIndex, i);
        avgPos[0] += point[0];
        avgPos[1] += point[1];
        avgPos[2] += point[2];
    }

    // total tex coords if needed
    if (geom.textureCoordinateIndexingMode != IndexingModeUndefined) {
        for (var i = 0; i < polygonPointCount; i++) {
            
            var texCoord = geom.getTextureCoordinateOnPolygon(polyIndex, i);
            avgTex[0] += texCoord[0];
            avgTex[1] += texCoord[1];
        }
    }

    // get the center 
    var div = 1.0 / polygonPointCount;
    avgPos[0] *= div;
    avgPos[1] *= div;
    avgPos[2] *= div;

    avgTex[0] *= div;
    avgTex[1] *= div;

    // add the avg point
    geom.addPoints(avgPos, 1);

    // store tex coords if (if we need to)
    // we'll put these into the geom later
    var texcoords = new Array();
    if (geom.textureCoordinateIndexingMode == IndexingModePerPoint) {
        geom.addTextureCoordinates(avgTex, 1);
    }
    else if (geom.textureCoordinateIndexingMode == IndexingModePerPointOnPolygon) {
        texcoords.push(avgTex);
    }

    // split points for this polygon
    var splitPointIndices = new Array();
    
    // compute the split points
    for (var i = 0; i < polygonPointCount; i++) {
        var i0 = i;
        var i1 = i + 1;
        if (i1 >= polygonPointCount) {
            i1 = 0;
        }
        var p0 = geom.getPointOnPolygon(polyIndex, i0);
        var p1 = geom.getPointOnPolygon(polyIndex, i1);

        // get object level indices
        var oi0 = geom.getPolygonPoint(polyIndex, i0);
        var oi1 = geom.getPolygonPoint(polyIndex, i1);

        // index of the point that splits this edge
        var splitPointIndex = 0;

        // we're splitting the edge between object level points
        // p0 and p1, so see if this is edge has been previously split

        var tmp0 = Math.min(oi0, oi1);
        var tmp1 = Math.max(oi0, oi1);

        // services.debug.trace("splitting edge: " + tmp0 + " " + tmp1);

        var wasNewPointCreated = false;
        if (splitEdgesMap.hasOwnProperty("" + tmp0) && (splitEdgesMap[tmp0].hasOwnProperty("" + tmp1)))
        {
            splitPointIndex = splitEdgesMap[tmp0][tmp1];
            // services.debug.trace("using split point: " + splitPointIndex);
        }
        else
        {
            wasNewPointCreated = true;

            var split = [0, 0, 0];
            split[0] = p0[0] + 0.5 * (p1[0] - p0[0]);
            split[1] = p0[1] + 0.5 * (p1[1] - p0[1]);
            split[2] = p0[2] + 0.5 * (p1[2] - p0[2]);

            if (splitEdgesMap.hasOwnProperty("" + tmp0) == false) {
                splitEdgesMap[tmp0] = new Object();
            }

            splitPointIndex = geom.pointCount;
            geom.addPoints(split, 1);

            // services.debug.trace("created split point: " + splitPointIndex);

            splitEdgesMap[tmp0][tmp1] = splitPointIndex;
        }
        splitPointIndices.push(splitPointIndex);

        if (geom.textureCoordinateIndexingMode != IndexingModeUndefined) {

            var texCoord0 = geom.getTextureCoordinateOnPolygon(polyIndex, i0);
            var texCoord1 = geom.getTextureCoordinateOnPolygon(polyIndex, i1);
            var texCoord = [0, 0];
            texCoord[0] = texCoord0[0] + 0.5 * (texCoord1[0] - texCoord0[0]);
            texCoord[1] = texCoord0[1] + 0.5 * (texCoord1[1] - texCoord0[1]);

            if (geom.textureCoordinateIndexingMode == IndexingModePerPoint) {
                if (wasNewPointCreated) {
                    geom.addTextureCoordinates(texCoord, 1);
                }   
            }
            else if (geom.textureCoordinateIndexingMode == IndexingModePerPointOnPolygon) {
                texcoords.push(texCoord);
            }
        }

        //
        // we need to split edges on polys if the edge is shared
        //
        var polygonsSharingEdge = geom.getPolygonsFromEdge(tmp0, tmp1);
        var sharedPolyCount = polygonsSharingEdge.getPolygonCount();
        for (var j = 0; j < sharedPolyCount; j++) {
            var sharedPolygonIndex = polygonsSharingEdge.getPolygon(j);
            if (polyCollection.hasPolygon(sharedPolygonIndex) == false) {
                // this is a shared poly, and not part of the set we're going to subdivide explicitly
                // so we need to break the edge tmp0, tmp1
                geom.insertPolygonPoint(sharedPolygonIndex, tmp0, tmp1, splitPointIndex);
            }
        }
    }

    // we now create a new polygon (quad) for each point in the old poly
    // i.e oldpoly point count == old poly edge count == number of new polys created after split
    for (var i = 0; i < polygonPointCount; i++) {

        // wrap around
        var prevIndex;
        if (i > 0) {
            prevIndex = i - 1;
        }
        else {
            prevIndex = polygonPointCount - 1;
        }

        // add a polygon
        var thisPolyIndex = geom.polygonCount;
        geom.addPolygon(materialIndex);

        polysToSelect.push(thisPolyIndex);

        // add points
        var i0 = geom.getPolygonPoint(polyIndex, i);
        var i1 = splitPointIndices[i];
        var i2 = newPointStart;
        var i3 = splitPointIndices[prevIndex];

        geom.addPolygonPoint(thisPolyIndex, i0);
        geom.addPolygonPoint(thisPolyIndex, i1);
        geom.addPolygonPoint(thisPolyIndex, i2);
        geom.addPolygonPoint(thisPolyIndex, i3);

        if (geom.textureCoordinateIndexingMode == IndexingModePerPointOnPolygon) {
            geom.addTextureCoordinates(geom.getTextureCoordinateOnPolygon(polyIndex, i), 1);
            geom.addTextureCoordinates(texcoords[i + 1], 1);
            geom.addTextureCoordinates(texcoords[0], 1);
            geom.addTextureCoordinates(texcoords[prevIndex + 1], 1);
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// helper to get a designer property as a bool
//
///////////////////////////////////////////////////////////////////////////////
function getDesignerPropAsBool(tname) {
    if (document.designerProps.hasTrait(tname))
        return document.designerProps.getTrait(tname).value;

    return false;
}

function getSelectionMode() {
    if (getDesignerPropAsBool("usePivot"))
        return 0; // default to object mode when using pivot
    if (document.designerProps.hasTrait("SelectionMode"))
        return document.designerProps.getTrait("SelectionMode").value;
    return 0;
}


// find the mesh child
function findFirstChildMeshElement(parent)
{
    // find the mesh child
    for (var i = 0; i < parent.childCount; i++) {

        // get child and its materials
        var child = parent.getChild(i);
        if (child.typeId == "Microsoft.VisualStudio.3D.Mesh") {
            return child;
        }
    }
    return null;
}


function UndoableItem(collElem, meshElem) {

    this._clonedColl = collElem.clone();
    this._polyCollection = this._clonedColl.behavior;
    this._meshElem = meshElem;
    this._mesh = meshElem.behavior;

    var geom = this._meshElem.getTrait("Geometry").value;
    this._restoreGeom = geom.clone();

    this.getName = function () {
        var IDS_MreUndoSubdivide = 147;
        return services.strings.getStringFromId(IDS_MreUndoSubdivide);
    }

    this.onDo = function () {

        var newCollection = this._clonedColl.clone();
        var newPolyBeh = newCollection.behavior;
        newPolyBeh.clear();

        // maps split edges to indices of points used to split them
        var splitEdgesMap = new Object();

        var geom = this._meshElem.getTrait("Geometry").value;

        var polysToSelect = new Array();

        // subdivide
        var polysToDelete = new Array();
        var polyCount = this._polyCollection.getPolygonCount();
        for (var i = 0; i < polyCount; i++) {
            var polyIndex = this._polyCollection.getPolygon(i);
            addSubdivisions(geom, this._polyCollection, polyIndex, splitEdgesMap, polysToSelect);

            polysToDelete.push(polyIndex);
        }

        function sortNumberDescending(a, b) {
            return b - a;
        }
 
        // delete the old selection
        polysToDelete.sort(sortNumberDescending);

        var numDeletedPolys = polysToDelete.length;

        for (var p = 0; p < polysToDelete.length; p++) {
            geom.removePolygon(polysToDelete[p]);
        }

        // shift polygon indices
        for (var p = 0; p < polysToSelect.length; p++) {
            var indexToSelect = polysToSelect[p] - numDeletedPolys;

            newPolyBeh.addPolygon(indexToSelect);
        }

        this._mesh.selectedObjects = newCollection;

        this._mesh.recomputeCachedGeometry();
    }

    this.onUndo = function () {
        var geom = this._meshElem.getTrait("Geometry").value;
        geom.copyFrom(this._restoreGeom);

        this._mesh.selectedObjects = this._clonedColl;

        this._mesh.recomputeCachedGeometry();
    }
}

var selectedElement = document.selectedElement;
var selectionMode = getSelectionMode();

// get the poly collection
var polyCollection = null;
var mesh = null;
var meshElem = null;
var collElem = null;
if (selectedElement != null) {
    if (selectionMode == 1) {
        meshElem = findFirstChildMeshElement(selectedElement);
        if (meshElem != null) {
            mesh = meshElem.behavior;

            // polygon selection mode
            collElem = mesh.selectedObjects;
            if (collElem != null) {
                polyCollection = collElem.behavior;
            }
        }
    }
}

if (polyCollection != null && collElem.typeId == "PolygonCollection") {
    var undoableItem = new UndoableItem(collElem, meshElem);
    undoableItem.onDo();
    services.undoService.addUndoableItem(undoableItem);
}
// SIG // Begin signature block
// SIG // MIIkCQYJKoZIhvcNAQcCoIIj+jCCI/YCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // vVb+u/YO+8d4nxqG8U1pDSpZo+x+WxGf5RzP6XhH35eg
// SIG // gg2TMIIGETCCA/mgAwIBAgITMwAAAI6HkaRXGl/KPgAA
// SIG // AAAAjjANBgkqhkiG9w0BAQsFADB+MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
// SIG // aWduaW5nIFBDQSAyMDExMB4XDTE2MTExNzIyMDkyMVoX
// SIG // DTE4MDIxNzIyMDkyMVowgYMxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xDTALBgNVBAsTBE1PUFIxHjAcBgNVBAMTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjCCASIwDQYJKoZIhvcNAQEB
// SIG // BQADggEPADCCAQoCggEBANCH1EIrfp3ZxnrUosPjFZLS
// SIG // U52VF8lSNvpUv6sQr+nJ58wmU8PCc79t1gDlANzpamc0
// SIG // MPYWF7QBpZV8i7mkLOaLO3n2Iwx5j/NS30ABHMLGA53r
// SIG // Wc9z6dhxOZvwziVZLdLJWwrvftYyDl10EgTsngRTpmsC
// SIG // Z/hNWYt34Csh4O/ApEUSzwN7A8Y5w9Qi3FVcd0L/nLLl
// SIG // VWdoui12an9mU0fVRwrMON6Ne5cZfYLQJviljuWh8F5k
// SIG // EOT56yfG8uAI0A3yZ8DY8i/7idoV+a4PPgCXB9ELPnDU
// SIG // d6tyeEGYB7gXzKKxX+y981Bno9eU8NKLVY9TppWT5rJm
// SIG // z8k3aORjx88CAwEAAaOCAYAwggF8MB8GA1UdJQQYMBYG
// SIG // CisGAQQBgjdMCAEGCCsGAQUFBwMDMB0GA1UdDgQWBBSr
// SIG // yNbtshXSqo7xzO1sOPdFStCKuzBSBgNVHREESzBJpEcw
// SIG // RTENMAsGA1UECxMETU9QUjE0MDIGA1UEBRMrMjMwMDEy
// SIG // K2IwNTBjNmU3LTc2NDEtNDQxZi1iYzRhLTQzNDgxZTQx
// SIG // NWQwODAfBgNVHSMEGDAWgBRIbmTlUAXTgqoXNzcitW2o
// SIG // ynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9NaWNDb2RT
// SIG // aWdQQ0EyMDExXzIwMTEtMDctMDguY3JsMGEGCCsGAQUF
// SIG // BwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNDb2RT
// SIG // aWdQQ0EyMDExXzIwMTEtMDctMDguY3J0MAwGA1UdEwEB
// SIG // /wQCMAAwDQYJKoZIhvcNAQELBQADggIBAESJAqxpU/PE
// SIG // trvUjGBT58psqElpZr6lmkGZOtid0lcCUWr6v5uW26Ym
// SIG // fQlW6NztJXV6pUdSqB5LFlPz7g+awwSVKcGChKRWMfyg
// SIG // ipGVtb9azqkBH2RGoebK8dd0e7+SCFFefDMCXlE7m+XY
// SIG // Ll8CTAmcGkPace3k2eei2nQsF63lDLUY9VQJ1L4cc80g
// SIG // e6T6yNvY2zqu+pDFo72VZa5GLVcpWNaS8GzaY/GPM6J+
// SIG // OHZe3fM17ayaO2KB0E4ZfEh8sAuPOMwtvNU5ZamVwQPi
// SIG // ksm5q9JXCqrcUgsuViej4piXV468qVluJJKOguIJc4LZ
// SIG // NYPMn3/RBI6IuOKag1iw1JrmMfqUR459puJOefPY02oz
// SIG // FlBw8UK7mAnp/8yVVVsIv5JSqAjE8ejx/0DX+Zo2nf26
// SIG // kIXSVT5QrUYf7yUMuJ46SARj73iYol0DDQLY3CCr5la1
// SIG // 3u8WZsPXVYIeT4J4yZ5UGhBgtxerQBORrrAZwZozne4y
// SIG // cs1lzE9GmC0PUWAefPv+2+gHeQf3oTM4/gma2497tjq9
// SIG // hYa4zLx9ATC3ex2pXRu9zE0X925HM9VA32rKLlG4tbnP
// SIG // wwTTO+Xj6RCM66e63qQuM2opLxRK6h7BIjg1BYXvwgQA
// SIG // DWvB2JYUSBWvflKwuGDEUrVKgreFKgBJKiaDJ1pB3r3V
// SIG // Zkm8C5x4cAm8MIIHejCCBWKgAwIBAgIKYQ6Q0gAAAAAA
// SIG // AzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMx
// SIG // EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
// SIG // ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
// SIG // dGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2Vy
// SIG // dGlmaWNhdGUgQXV0aG9yaXR5IDIwMTEwHhcNMTEwNzA4
// SIG // MjA1OTA5WhcNMjYwNzA4MjEwOTA5WjB+MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29k
// SIG // ZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAg8AMIICCgKCAgEAq/D6chAcLq3YbqqCEE00
// SIG // uvK2WCGfQhsqa+laUKq4BjgaBEm6f8MMHt03a8YS2Avw
// SIG // OMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe0t+b
// SIG // U7IKLMOv2akrrnoJr9eWWcpgGgXpZnboMlImEi/nqwhQ
// SIG // z7NEt13YxC4Ddato88tt8zpcoRb0RrrgOGSsbmQ1eKag
// SIG // Yw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v++MrWhAf
// SIG // TVYoonpy4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+E
// SIG // GvKhL1nkkDstrjNYxbc+/jLTswM9sbKvkjh+0p2ALPVO
// SIG // VpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN91/w0FK/jJSH
// SIG // vMAhdCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rh
// SIG // KEmdX4jiJV3TIUs+UsS1Vz8kA/DRelsv1SPjcF0PUUZ3
// SIG // s/gA4bysAoJf28AVs70b1FVL5zmhD+kjSbwYuER8ReTB
// SIG // w3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zd
// SIG // sGbiwZeBe+3W7UvnSSmnEyimp31ngOaKYnhfsi+E11ec
// SIG // XL93KCjx7W3DKI8sj0A3T8HhhUSJxAlMxdSlQy90lfdu
// SIG // +HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaI
// SIG // jAsCAwEAAaOCAe0wggHpMBAGCSsGAQQBgjcVAQQDAgEA
// SIG // MB0GA1UdDgQWBBRIbmTlUAXTgqoXNzcitW2oynUClTAZ
// SIG // BgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8E
// SIG // BAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAW
// SIG // gBRyLToCMZBDuRQFTuHqp8cx0SOJNDBaBgNVHR8EUzBR
// SIG // ME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
// SIG // cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDEx
// SIG // XzIwMTFfMDNfMjIuY3JsMF4GCCsGAQUFBwEBBFIwUDBO
// SIG // BggrBgEFBQcwAoZCaHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIw
// SIG // MTFfMDNfMjIuY3J0MIGfBgNVHSAEgZcwgZQwgZEGCSsG
// SIG // AQQBgjcuAzCBgzA/BggrBgEFBQcCARYzaHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1h
// SIG // cnljcHMuaHRtMEAGCCsGAQUFBwICMDQeMiAdAEwAZQBn
// SIG // AGEAbABfAHAAbwBsAGkAYwB5AF8AcwB0AGEAdABlAG0A
// SIG // ZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oal
// SIG // mOBUeRou09h0ZyKbC5YR4WOSmUKWfdJ5DJDBZV8uLD74
// SIG // w3LRbYP+vj/oCso7v0epo/Np22O/IjWll11lhJB9i0ZQ
// SIG // VdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1HXeU
// SIG // OeLpZMlEPXh6I/MTfaaQdION9MsmAkYqwooQu6SpBQyb
// SIG // 7Wj6aC6VoCo/KmtYSWMfCWluWpiW5IP0wI/zRive/DvQ
// SIG // vTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInWH8MyGOLw
// SIG // xS3OW560STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn
// SIG // +N4sOiBpmLJZiWhub6e3dMNABQamASooPoI/E01mC8Cz
// SIG // TfXhj38cbxV9Rad25UAqZaPDXVJihsMdYzaXht/a8/jy
// SIG // FqGaJ+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw
// SIG // 3MYbBL7fQccOKO7eZS/sl/ahXJbYANahRr1Z85elCUtI
// SIG // EJmAH9AAKcWxm6U/RXceNcbSoqKfenoi+kiVH6v7RyOA
// SIG // 9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sL
// SIG // gOppO6/8MO0ETI7f33VtY5E90Z1WTk+/gFcioXgRMiF6
// SIG // 70EKsT/7qMykXcGhiJtXcVZOSEXAQsmbdlsKgEhr/Xmf
// SIG // wb1tbWrJUnMTDXpQzTGCFc4wghXKAgEBMIGVMH4xCzAJ
// SIG // BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAw
// SIG // DgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3Nv
// SIG // ZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29m
// SIG // dCBDb2RlIFNpZ25pbmcgUENBIDIwMTECEzMAAACOh5Gk
// SIG // Vxpfyj4AAAAAAI4wDQYJYIZIAWUDBAIBBQCggbwwGQYJ
// SIG // KoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
// SIG // NwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkE
// SIG // MSIEIIAGLOI8qMuSAI9Msp8JMbWb90tOHn+Qm9EmhZuj
// SIG // X6nHMFAGCisGAQQBgjcCAQwxQjBAoCaAJABBAGQAZABT
// SIG // AHUAYgBkAGkAdgBpAHMAaQBvAG4AcwAuAGoAc6EWgBRo
// SIG // dHRwOi8vbWljcm9zb2Z0LmNvbTANBgkqhkiG9w0BAQEF
// SIG // AASCAQA5zD87iQ4nbSGNHiPXhyBgRrItc/NcRCV008dX
// SIG // 3DroQn/l8xbLTba0XmnUt31zlsS7gJY6Q75AovWcGAXl
// SIG // EWVkCDpk3gP7dRG1nTxZ5DNLjc7ySklPZ39Exx3wSDx6
// SIG // lX1PxUHLKbWBi8PY3ke3V/wLcH4eTkpzHq7ecNfOKuwV
// SIG // UxpgCf3E0MM22aJX1WS+nD/RdC2Ru24fC/9dNNiHT3uz
// SIG // MOrUA/97J5H17OO0jmrWYW/IC5zH4H4VQBRzNiqATwiD
// SIG // +2KTei68/mB7NhB/B3rLuXUi70t6XfQShIO5R8M4cLHP
// SIG // VpINOs7wevlJzq24isQ7s2gc9ujvqBlXi8DzfXaNoYIT
// SIG // SjCCE0YGCisGAQQBgjcDAwExghM2MIITMgYJKoZIhvcN
// SIG // AQcCoIITIzCCEx8CAQMxDzANBglghkgBZQMEAgEFADCC
// SIG // AT0GCyqGSIb3DQEJEAEEoIIBLASCASgwggEkAgEBBgor
// SIG // BgEEAYRZCgMBMDEwDQYJYIZIAWUDBAIBBQAEIM5hvpvb
// SIG // NGhu7x2fkBWjKhL+IZQi7LoQNmXuxNwlE9LzAgZYr6X4
// SIG // JRMYEzIwMTcwMzA0MjMyNDA2Ljg5M1owBwIBAYACAfSg
// SIG // gbmkgbYwgbMxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
// SIG // YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
// SIG // VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xDTALBgNV
// SIG // BAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIgRFNFIEVT
// SIG // TjpGNTI4LTM3NzctOEE3NjElMCMGA1UEAxMcTWljcm9z
// SIG // b2Z0IFRpbWUtU3RhbXAgU2VydmljZaCCDs0wggZxMIIE
// SIG // WaADAgECAgphCYEqAAAAAAACMA0GCSqGSIb3DQEBCwUA
// SIG // MIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylN
// SIG // aWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3Jp
// SIG // dHkgMjAxMDAeFw0xMDA3MDEyMTM2NTVaFw0yNTA3MDEy
// SIG // MTQ2NTVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
// SIG // YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
// SIG // VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNV
// SIG // BAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
// SIG // MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
// SIG // qR0NvHcRijog7PwTl/X6f2mUa3RUENWlCgCChfvtfGhL
// SIG // LF/Fw+Vhwna3PmYrW/AVUycEMR9BGxqVHc4JE458YTBZ
// SIG // sTBED/FgiIRUQwzXTbg4CLNC3ZOs1nMwVyaCo0UN0Or1
// SIG // R4HNvyRgMlhgRvJYR4YyhB50YWeRX4FUsc+TTJLBxKZd
// SIG // 0WETbijGGvmGgLvfYfxGwScdJGcSchohiq9LZIlQYrFd
// SIG // /XcfPfBXday9ikJNQFHRD5wGPmd/9WbAA5ZEfu/QS/1u
// SIG // 5ZrKsajyeioKMfDaTgaRtogINeh4HLDpmc085y9Euqf0
// SIG // 3GS9pAHBIAmTeM38vMDJRF1eFpwBBU8iTQIDAQABo4IB
// SIG // 5jCCAeIwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYE
// SIG // FNVjOlyKMZDzQ3t8RhvFM2hahW1VMBkGCSsGAQQBgjcU
// SIG // AgQMHgoAUwB1AGIAQwBBMAsGA1UdDwQEAwIBhjAPBgNV
// SIG // HRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFNX2VsuP6KJc
// SIG // YmjRPZSQW9fOmhjEMFYGA1UdHwRPME0wS6BJoEeGRWh0
// SIG // dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3By
// SIG // b2R1Y3RzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNy
// SIG // bDBaBggrBgEFBQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0
// SIG // dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMv
// SIG // TWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3J0MIGgBgNV
// SIG // HSABAf8EgZUwgZIwgY8GCSsGAQQBgjcuAzCBgTA9Bggr
// SIG // BgEFBQcCARYxaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
// SIG // L1BLSS9kb2NzL0NQUy9kZWZhdWx0Lmh0bTBABggrBgEF
// SIG // BQcCAjA0HjIgHQBMAGUAZwBhAGwAXwBQAG8AbABpAGMA
// SIG // eQBfAFMAdABhAHQAZQBtAGUAbgB0AC4gHTANBgkqhkiG
// SIG // 9w0BAQsFAAOCAgEAB+aIUQ3ixuCYP4FxAz2do6Ehb7Pr
// SIG // psz1Mb7PBeKp/vpXbRkws8LFZslq3/Xn8Hi9x6ieJeP5
// SIG // vO1rVFcIK1GCRBL7uVOMzPRgEop2zEBAQZvcXBf/XPle
// SIG // FzWYJFZLdO9CEMivv3/Gf/I3fVo/HPKZeUqRUgCvOA8X
// SIG // 9S95gWXZqbVr5MfO9sp6AG9LMEQkIjzP7QOllo9ZKby2
// SIG // /QThcJ8ySif9Va8v/rbljjO7Yl+a21dA6fHOmWaQjP9q
// SIG // Yn/dxUoLkSbiOewZSnFjnXshbcOco6I8+n99lmqQeKZt
// SIG // 0uGc+R38ONiU9MalCpaGpL2eGq4EQoO4tYCbIjggtSXl
// SIG // ZOz39L9+Y1klD3ouOVd2onGqBooPiRa6YacRy5rYDkea
// SIG // gMXQzafQ732D8OE7cQnfXXSYIghh2rBQHm+98eEA3+cx
// SIG // B6STOvdlR3jo+KhIq/fecn5ha293qYHLpwmsObvsxsvY
// SIG // grRyzR30uIUBHoD7G4kqVDmyW9rIDVWZeodzOwjmmC3q
// SIG // jeAzLhIp9cAvVCch98isTtoouLGp25ayp0Kiyc8ZQU3g
// SIG // hvkqmqMRZjDTu3QyS99je/WZii8bxyGvWbWu3EQ8l1Bx
// SIG // 16HSxVXjad5XwdHeMMD9zOZN+w2/XU/pnR4ZOC+8z1gF
// SIG // Lu8NoFA12u8JJxzVs341Hgi62jbb01+P3nSISRIwggTa
// SIG // MIIDwqADAgECAhMzAAAAsIbpkiAKXer4AAAAAACwMA0G
// SIG // CSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // JjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBD
// SIG // QSAyMDEwMB4XDTE2MDkwNzE3NTY1NloXDTE4MDkwNzE3
// SIG // NTY1NlowgbMxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
// SIG // YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
// SIG // VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xDTALBgNV
// SIG // BAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIgRFNFIEVT
// SIG // TjpGNTI4LTM3NzctOEE3NjElMCMGA1UEAxMcTWljcm9z
// SIG // b2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIwDQYJKoZI
// SIG // hvcNAQEBBQADggEPADCCAQoCggEBAPDl8I2UWama12wJ
// SIG // B3cUIVldtz14uDWt9TAx/87h/9bpGsmDmCP7kBDBkuw6
// SIG // vFBskL1c8WutTKL1YaWH4SmsozwS33/EAILOvbQKX8u5
// SIG // 25q3GS1a7QcArVT9M6z2+BIkSiRnV9q4S6GsD4D5Ob0g
// SIG // DbWnxCxL8aFW5vIvJ7kmwTMl2SsMMHJQhGuSapWM8voI
// SIG // TbguJNVD09zr83RqiLPJtGDcyWWtXwnlEWoUNMHkVwYS
// SIG // RuagvYRaJEJD9rz0LFWZWo2ziKaqEpQivoa1SomMqkew
// SIG // oMUfG4+c8GJtVJ0y06nEm1PpuXI7vxeer1hcncYrUCG7
// SIG // 2JbMNu+NpnPZrbtDcBUCAwEAAaOCARswggEXMB0GA1Ud
// SIG // DgQWBBT8qAdS2e8p/phDPBVcdF63f3PgrzAfBgNVHSME
// SIG // GDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNUaW1TdGFQQ0Ff
// SIG // MjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY1RpbVN0YVBDQV8yMDEwLTA3
// SIG // LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoG
// SIG // CCsGAQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQCS/YD3
// SIG // db1lkD4XzdUVg8WOApE7rE59YO+POQaTkOMPqkU9iT8C
// SIG // /exvuFCQCWyE0jchmbHBhnYd5A3Gjv8Y4KL6RrWFd3Ru
// SIG // TgxERdC80rYEfnBLD9arB2lWSnZk6PmqR6Ojhtz0z0S4
// SIG // SRBbZ4/R2nmuOt2HUwboj2r8f9fko4ZFFQ2RYiI6kErj
// SIG // 2GgBScktDmt4bBEC9I9NEWBzqAMhDZfjD3J1u/3oDrjr
// SIG // /CBBnY6T/cUzcz45er1qNB45R2IRtRzvV/Rzg3GfqBRl
// SIG // vuQDjk1txAEzG35X7KHzHJepOKf0qDo5+Yk1/F2jIq4K
// SIG // vY0MqzESbum9YRbs9Bk6JIViX6myoYIDdjCCAl4CAQEw
// SIG // geOhgbmkgbYwgbMxCzAJBgNVBAYTAlVTMRMwEQYDVQQI
// SIG // EwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4w
// SIG // HAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xDTAL
// SIG // BgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIgRFNF
// SIG // IEVTTjpGNTI4LTM3NzctOEE3NjElMCMGA1UEAxMcTWlj
// SIG // cm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZaIlCgEBMAkG
// SIG // BSsOAwIaBQADFQC8hPudWxLaxzaFO4hmnqMWGNXMHKCB
// SIG // wjCBv6SBvDCBuTELMAkGA1UEBhMCVVMxEzARBgNVBAgT
// SIG // Cldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAc
// SIG // BgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjENMAsG
// SIG // A1UECxMETU9QUjEnMCUGA1UECxMebkNpcGhlciBOVFMg
// SIG // RVNOOjU3RjYtQzFFMC01NTRDMSswKQYDVQQDEyJNaWNy
// SIG // b3NvZnQgVGltZSBTb3VyY2UgTWFzdGVyIENsb2NrMA0G
// SIG // CSqGSIb3DQEBBQUAAgUA3GVwtzAiGA8yMDE3MDMwNDE2
// SIG // NTgzMVoYDzIwMTcwMzA1MTY1ODMxWjB0MDoGCisGAQQB
// SIG // hFkKBAExLDAqMAoCBQDcZXC3AgEAMAcCAQACAiP5MAcC
// SIG // AQACAhjyMAoCBQDcZsI3AgEAMDYGCisGAQQBhFkKBAIx
// SIG // KDAmMAwGCisGAQQBhFkKAwGgCjAIAgEAAgMW42ChCjAI
// SIG // AgEAAgMHoSAwDQYJKoZIhvcNAQEFBQADggEBACAwOeRf
// SIG // Pe/4le/CuUf42nC4Q8mpYXSyYC4kgdx7XTAStxtJStSu
// SIG // vJHSEWcPZWV8ubngq2t84C9Naad5DGIYuSnjmuLUZoq1
// SIG // huagVeIxzzJ0y0GOGHSxAqoDtM901pQ5QN7WrXXpIvyu
// SIG // jBwdodNSKeblETSu+Nkf+AlkM7Hb513gLX/s60r6wo4x
// SIG // 4hSNqdIKyI6ox6H5iAJHKKmct58VUVHBE4CbqeVUVbTA
// SIG // CD9+w3NZ35j+W0x2nuzDI5JShHMOypK3cN4/ftlHBSdq
// SIG // 7wO78kTGiMKTUS62wk9v5C/6BsRNmAV//0SiuV/ocbrk
// SIG // zZYkIHm13U2R3OGpYWCsqKMkXzgxggL1MIIC8QIBATCB
// SIG // kzB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1N
// SIG // aWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAA
// SIG // ALCG6ZIgCl3q+AAAAAAAsDANBglghkgBZQMEAgEFAKCC
// SIG // ATIwGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8G
// SIG // CSqGSIb3DQEJBDEiBCAOE4ZDVhUFN+STyvwOu5CIrCwb
// SIG // xvVSD8GKcel1M0u+CzCB4gYLKoZIhvcNAQkQAgwxgdIw
// SIG // gc8wgcwwgbEEFLyE+51bEtrHNoU7iGaeoxYY1cwcMIGY
// SIG // MIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
// SIG // c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
// SIG // BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UE
// SIG // AxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAC
// SIG // EzMAAACwhumSIApd6vgAAAAAALAwFgQUHDxJnoC5FXFR
// SIG // Ren1k7rNpGE/VScwDQYJKoZIhvcNAQELBQAEggEAMM5r
// SIG // odr0IxKzqT6nArjDzKxt0CO4SywSC8JAMPpw0R0Rdwn/
// SIG // V5+2kG3osXperSnW6nKkwWlgyUmgElvbes2sPVOFnaYP
// SIG // JxBo86PQItbfBwDkIrmzLd8IZAhV+7sjwkbS/t9TX/Um
// SIG // 0iMM25US4NgHn4F+bDw7/NnmjyJq2S99w0qNKc+Y4Vu7
// SIG // P+HLrgD/PmY5PGfXccuHSgezsXpK+DfYAyoLSfdXtbwH
// SIG // o5VSFb7t4jMsZEFF50pwBKolCJushEyqdrY5ya7P73iS
// SIG // eBTpWzyJArbktSg/57yNHBCx4aHp+qSmqOnlC2l9HTxj
// SIG // dlFxcyCAhXI7cpS3ZT0uU+ZHbUOtzA==
// SIG // End signature block
